import os
import argparse

PROGRAM_EXTENSIONS = {'.py', '.js', '.java', '.c', '.cpp', '.h', '.hpp', '.cs', '.php', '.rb', '.go', '.rs', '.swift', '.kt', '.ts', '.jsx', '.tsx', '.vue', '.html', '.css', '.scss', '.sass', '.less', '.json', '.xml', '.yaml', '.yml', '.sh', '.bat', '.ps1', '.sql', '.r', '.m', '.lua', '.pl', '.pm', '.scala', '.groovy', '.dart', '.jsx', '.tsx'}

def is_valid_file(filepath):
    ext = os.path.splitext(filepath)[1].lower()
    return ext == '.txt' or ext in PROGRAM_EXTENSIONS

def remove_empty_lines(filepath):
    if not is_valid_file(filepath):
        print(f"跳过不支持的文件: {filepath}")
        return False
    
    with open(filepath, 'r', encoding='utf-8') as f:
        lines = f.readlines()
    
    non_empty_lines = [line for line in lines if line.strip()]
    
    if len(non_empty_lines) == len(lines):
        print(f"无需处理: {filepath}")
        return False
    
    with open(filepath, 'w', encoding='utf-8') as f:
        f.writelines(non_empty_lines)
    
    removed_count = len(lines) - len(non_empty_lines)
    print(f"已处理: {filepath} (移除 {removed_count} 个空行)")
    return True

def process_file(filepath):
    if os.path.isfile(filepath):
        remove_empty_lines(filepath)
    else:
        print(f"错误: 文件不存在 - {filepath}")

def process_directory(directory):
    if not os.path.isdir(directory):
        print(f"错误: 目录不存在 - {directory}")
        return
    
    processed_count = 0
    for root, dirs, files in os.walk(directory):
        for filename in files:
            filepath = os.path.join(root, filename)
            if remove_empty_lines(filepath):
                processed_count += 1
    
    print(f"\n处理完成! 共处理 {processed_count} 个文件")

def main():
    parser = argparse.ArgumentParser(description='批量去除文件空行工具')
    parser.add_argument('path', help='文件或目录路径')
    args = parser.parse_args()
    
    path = args.path
    
    if not os.path.exists(path):
        print(f"错误: 路径不存在 - {path}")
        return
    
    print(f"开始处理: {path}")
    print("-" * 50)
    
    if os.path.isfile(path):
        process_file(path)
    elif os.path.isdir(path):
        process_directory(path)
    
    print("-" * 50)
    print("处理完成!")

if __name__ == '__main__':
    main()
