import os
import json
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext
from tkinter import ttk
from datetime import datetime

PROGRAM_EXTENSIONS = {'.py', '.js', '.java', '.c', '.cpp', '.h', '.hpp', '.cs', '.php', '.rb', '.go', '.rs', '.swift', '.kt', '.ts', '.jsx', '.tsx', '.vue', '.html', '.css', '.scss', '.sass', '.less', '.json', '.xml', '.yaml', '.yml', '.sh', '.bat', '.ps1', '.sql', '.r', '.m', '.lua', '.pl', '.pm', '.scala', '.groovy', '.dart'}
HISTORY_FILE = "history.json"

class RemoveEmptyLinesApp:
    def __init__(self, root):
        self.root = root
        self.root.title("批量去空行工具 v2.0 - 冰橙文化科技 出品 www.bcwhkj.cn")
        self.root.geometry("900x650")
        self.root.resizable(True, True)
        
        self.history = self.load_history()
        self.setup_styles()
        self.setup_ui()
    
    def setup_styles(self):
        style = ttk.Style()
        style.theme_use('clam')
        
        style.configure('Title.TLabel', font=('Microsoft YaHei UI', 12, 'bold'), foreground='#2c3e50')
        style.configure('Subtitle.TLabel', font=('Microsoft YaHei UI', 10, 'bold'), foreground='#34495e')
        style.configure('Normal.TLabel', font=('Microsoft YaHei UI', 9), foreground='#555555')
        
        style.configure('TButton', font=('Microsoft YaHei UI', 9), padding=8)
        style.configure('Primary.TButton', font=('Microsoft YaHei UI', 9, 'bold'), padding=10)
        style.map('Primary.TButton', background=[('active', '#2980b9'), ('pressed', '#1a5276')])
        
        style.configure('TFrame', background='#f8f9fa')
        style.configure('TLabelframe', background='#f8f9fa', borderwidth=1)
        style.configure('TLabelframe.Label', font=('Microsoft YaHei UI', 9, 'bold'), foreground='#2c3e50')
        
        style.configure('TEntry', fieldbackground='white', font=('Microsoft YaHei UI', 9))
        style.configure('TProgressbar', thickness=20, troughcolor='#e0e0e0')
        
        style.configure('TNotebook', background='#f8f9fa')
        style.configure('TNotebook.Tab', font=('Microsoft YaHei UI', 9), padding=[15, 8])
        
        self.root.configure(bg='#f8f9fa')
    
    def setup_ui(self):
        main_container = tk.Frame(self.root, bg='#f8f9fa')
        main_container.pack(fill=tk.BOTH, expand=True, padx=15, pady=15)
        
        title_frame = tk.Frame(main_container, bg='#3498db', height=60)
        title_frame.pack(fill=tk.X, pady=(0, 15))
        title_frame.pack_propagate(False)
        
        title_label = tk.Label(title_frame, text="批量去空行工具", font=('Microsoft YaHei UI', 18, 'bold'), 
                              bg='#3498db', fg='white')
        title_label.pack(pady=15)
        
        content_frame = tk.Frame(main_container, bg='#f8f9fa')
        content_frame.pack(fill=tk.BOTH, expand=True)
        
        left_panel = tk.Frame(content_frame, bg='white', relief=tk.RAISED, bd=1)
        left_panel.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 8))
        
        right_panel = tk.Frame(content_frame, bg='white', relief=tk.RAISED, bd=1, width=250)
        right_panel.pack(side=tk.RIGHT, fill=tk.Y, padx=(8, 0))
        right_panel.pack_propagate(False)
        
        self.setup_left_panel(left_panel)
        self.setup_right_panel(right_panel)
    
    def setup_left_panel(self, parent):
        inner_frame = tk.Frame(parent, bg='white')
        inner_frame.pack(fill=tk.BOTH, expand=True, padx=15, pady=15)
        
        tk.Label(inner_frame, text="选择文件或目录", font=('Microsoft YaHei UI', 11, 'bold'), 
                bg='white', fg='#2c3e50').pack(anchor=tk.W, pady=(0, 10))
        
        path_frame = tk.Frame(inner_frame, bg='white')
        path_frame.pack(fill=tk.X, pady=(0, 15))
        
        self.path_var = tk.StringVar()
        path_entry = tk.Entry(path_frame, textvariable=self.path_var, font=('Microsoft YaHei UI', 9),
                             relief=tk.FLAT, bd=1, highlightthickness=1, highlightbackground='#bdc3c7',
                             highlightcolor='#3498db')
        path_entry.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(0, 8))
        
        button_frame = tk.Frame(inner_frame, bg='white')
        button_frame.pack(fill=tk.X, pady=(0, 15))
        
        file_btn = tk.Button(button_frame, text="📁 选择文件", command=self.select_file,
                            bg='#3498db', fg='white', font=('Microsoft YaHei UI', 9, 'bold'),
                            relief=tk.FLAT, padx=20, pady=8, cursor='hand2',
                            activebackground='#2980b9', activeforeground='white')
        file_btn.pack(side=tk.LEFT, padx=(0, 8))
        
        dir_btn = tk.Button(button_frame, text="📂 选择目录", command=self.select_directory,
                           bg='#2ecc71', fg='white', font=('Microsoft YaHei UI', 9, 'bold'),
                           relief=tk.FLAT, padx=20, pady=8, cursor='hand2',
                           activebackground='#27ae60', activeforeground='white')
        dir_btn.pack(side=tk.LEFT, padx=(0, 8))
        
        process_btn = tk.Button(button_frame, text="▶ 开始处理", command=self.process,
                              bg='#e74c3c', fg='white', font=('Microsoft YaHei UI', 9, 'bold'),
                              relief=tk.FLAT, padx=20, pady=8, cursor='hand2',
                              activebackground='#c0392b', activeforeground='white')
        process_btn.pack(side=tk.LEFT, padx=(0, 8))
        
        clear_btn = tk.Button(button_frame, text="🗑 清空日志", command=self.clear_log,
                            bg='#95a5a6', fg='white', font=('Microsoft YaHei UI', 9, 'bold'),
                            relief=tk.FLAT, padx=20, pady=8, cursor='hand2',
                            activebackground='#7f8c8d', activeforeground='white')
        clear_btn.pack(side=tk.LEFT)
        
        tk.Label(inner_frame, text="处理日志", font=('Microsoft YaHei UI', 11, 'bold'), 
                bg='white', fg='#2c3e50').pack(anchor=tk.W, pady=(10, 5))
        
        log_frame = tk.Frame(inner_frame, bg='#ecf0f1', relief=tk.FLAT, bd=1)
        log_frame.pack(fill=tk.BOTH, expand=True, pady=(0, 10))
        
        self.log_text = scrolledtext.ScrolledText(log_frame, wrap=tk.WORD, font=('Consolas', 9),
                                                 bg='#2c3e50', fg='#ecf0f1', relief=tk.FLAT,
                                                 insertbackground='#ecf0f1', selectbackground='#3498db')
        self.log_text.pack(fill=tk.BOTH, expand=True, padx=5, pady=5)
        
        self.progress_var = tk.DoubleVar()
        progress_frame = tk.Frame(inner_frame, bg='white')
        progress_frame.pack(fill=tk.X, pady=(5, 10))
        
        tk.Label(progress_frame, text="进度:", font=('Microsoft YaHei UI', 9), bg='white', fg='#555').pack(side=tk.LEFT)
        
        progress = ttk.Progressbar(progress_frame, variable=self.progress_var, maximum=100, 
                                   style='TProgressbar')
        progress.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(8, 0))
        
        self.status_var = tk.StringVar(value="就绪")
        status_label = tk.Label(inner_frame, textvariable=self.status_var, 
                               font=('Microsoft YaHei UI', 9), bg='#ecf0f1', fg='#2c3e50',
                               relief=tk.SUNKEN, anchor=tk.W, padx=10, pady=5)
        status_label.pack(fill=tk.X)
    
    def setup_right_panel(self, parent):
        inner_frame = tk.Frame(parent, bg='white')
        inner_frame.pack(fill=tk.BOTH, expand=True, padx=15, pady=15)
        
        tk.Label(inner_frame, text="📜 历史记录", font=('Microsoft YaHei UI', 11, 'bold'), 
                bg='white', fg='#2c3e50').pack(anchor=tk.W, pady=(0, 10))
        
        history_frame = tk.Frame(inner_frame, bg='#ecf0f1', relief=tk.FLAT)
        history_frame.pack(fill=tk.BOTH, expand=True, pady=(0, 10))
        
        scrollbar = tk.Scrollbar(history_frame, bg='#ecf0f1')
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        self.history_listbox = tk.Listbox(history_frame, font=('Microsoft YaHei UI', 9),
                                         bg='white', fg='#2c3e50', relief=tk.FLAT,
                                         yscrollcommand=scrollbar.set, selectbackground='#3498db',
                                         selectforeground='white', activestyle='none')
        self.history_listbox.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=5, pady=5)
        scrollbar.config(command=self.history_listbox.yview)
        
        self.history_listbox.bind('<Double-Button-1>', self.on_history_double_click)
        
        btn_frame = tk.Frame(inner_frame, bg='white')
        btn_frame.pack(fill=tk.X)
        
        use_btn = tk.Button(btn_frame, text="✓ 使用选中", command=self.use_selected_history,
                           bg='#3498db', fg='white', font=('Microsoft YaHei UI', 9),
                           relief=tk.FLAT, padx=15, pady=6, cursor='hand2',
                           activebackground='#2980b9', activeforeground='white')
        use_btn.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(0, 5))
        
        clear_history_btn = tk.Button(btn_frame, text="✕ 清空历史", command=self.clear_history,
                                     bg='#e74c3c', fg='white', font=('Microsoft YaHei UI', 9),
                                     relief=tk.FLAT, padx=15, pady=6, cursor='hand2',
                                     activebackground='#c0392b', activeforeground='white')
        clear_history_btn.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(5, 0))
        
        self.refresh_history_list()
    
    def load_history(self):
        try:
            if os.path.exists(HISTORY_FILE):
                with open(HISTORY_FILE, 'r', encoding='utf-8') as f:
                    return json.load(f)
        except:
            pass
        return []
    
    def save_history(self):
        try:
            with open(HISTORY_FILE, 'w', encoding='utf-8') as f:
                json.dump(self.history, f, ensure_ascii=False, indent=2)
        except Exception as e:
            self.log(f"保存历史记录失败: {str(e)}")
    
    def add_to_history(self, path, file_count, removed_lines):
        record = {
            'path': path,
            'timestamp': datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
            'file_count': file_count,
            'removed_lines': removed_lines,
            'type': '目录' if os.path.isdir(path) else '文件'
        }
        
        self.history.insert(0, record)
        if len(self.history) > 50:
            self.history = self.history[:50]
        
        self.save_history()
        self.refresh_history_list()
    
    def refresh_history_list(self):
        self.history_listbox.delete(0, tk.END)
        for record in self.history:
            display_text = f"[{record['type']}] {os.path.basename(record['path'])}"
            self.history_listbox.insert(tk.END, display_text)
    
    def on_history_double_click(self, event):
        self.use_selected_history()
    
    def use_selected_history(self):
        selection = self.history_listbox.curselection()
        if not selection:
            messagebox.showinfo("提示", "请先选择一条历史记录")
            return
        
        index = selection[0]
        record = self.history[index]
        self.path_var.set(record['path'])
        self.log(f"从历史记录选择: {record['path']}")
    
    def clear_history(self):
        if not self.history:
            messagebox.showinfo("提示", "历史记录为空")
            return
        
        if messagebox.askyesno("确认", "确定要清空所有历史记录吗？"):
            self.history = []
            self.save_history()
            self.refresh_history_list()
            self.log("已清空历史记录")
    
    def log(self, message):
        timestamp = datetime.now().strftime('%H:%M:%S')
        self.log_text.insert(tk.END, f"[{timestamp}] {message}\n")
        self.log_text.see(tk.END)
        self.root.update()
    
    def clear_log(self):
        self.log_text.delete(1.0, tk.END)
    
    def select_file(self):
        filepath = filedialog.askopenfilename(
            title="选择文件",
            filetypes=[
                ("程序文件", "*.py *.js *.java *.c *.cpp *.h *.hpp *.cs *.php *.rb *.go *.rs *.swift *.kt *.ts *.jsx *.tsx *.vue *.html *.css *.json *.xml *.yaml *.yml"),
                ("文本文件", "*.txt"),
                ("所有文件", "*.*")
            ]
        )
        if filepath:
            self.path_var.set(filepath)
            self.log(f"已选择文件: {filepath}")
    
    def select_directory(self):
        directory = filedialog.askdirectory(title="选择目录")
        if directory:
            self.path_var.set(directory)
            self.log(f"已选择目录: {directory}")
    
    def is_valid_file(self, filepath):
        ext = os.path.splitext(filepath)[1].lower()
        return ext == '.txt' or ext in PROGRAM_EXTENSIONS
    
    def remove_empty_lines(self, filepath):
        if not self.is_valid_file(filepath):
            return False, 0
        
        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                lines = f.readlines()
        except UnicodeDecodeError:
            try:
                with open(filepath, 'r', encoding='gbk') as f:
                    lines = f.readlines()
            except Exception as e:
                self.log(f"读取文件失败: {filepath} - {str(e)}")
                return False, 0
        
        non_empty_lines = [line for line in lines if line.strip()]
        
        if len(non_empty_lines) == len(lines):
            return False, 0
        
        try:
            with open(filepath, 'w', encoding='utf-8') as f:
                f.writelines(non_empty_lines)
        except Exception as e:
            self.log(f"写入文件失败: {filepath} - {str(e)}")
            return False, 0
        
        removed_count = len(lines) - len(non_empty_lines)
        return True, removed_count
    
    def process_file(self, filepath):
        self.log(f"正在处理文件: {filepath}")
        success, removed_count = self.remove_empty_lines(filepath)
        
        if success:
            self.log(f"  ✓ 已处理: 移除 {removed_count} 个空行")
            return 1, removed_count
        elif self.is_valid_file(filepath):
            self.log(f"  - 无需处理")
        else:
            self.log(f"  × 跳过不支持的文件类型")
        
        return 0, 0
    
    def process_directory(self, directory):
        self.log(f"正在扫描目录: {directory}")
        
        file_list = []
        for root, dirs, files in os.walk(directory):
            for filename in files:
                filepath = os.path.join(root, filename)
                if self.is_valid_file(filepath):
                    file_list.append(filepath)
        
        total_files = len(file_list)
        if total_files == 0:
            self.log("未找到可处理的文件")
            return 0, 0
        
        self.log(f"找到 {total_files} 个可处理文件")
        
        processed_count = 0
        total_removed = 0
        for i, filepath in enumerate(file_list):
            self.status_var.set(f"正在处理: {i+1}/{total_files}")
            self.progress_var.set((i / total_files) * 100)
            self.root.update()
            
            success, removed_count = self.remove_empty_lines(filepath)
            if success:
                self.log(f"  ✓ {os.path.basename(filepath)}: 移除 {removed_count} 个空行")
                processed_count += 1
                total_removed += removed_count
            else:
                self.log(f"  - {os.path.basename(filepath)}: 无需处理")
        
        self.progress_var.set(100)
        return processed_count, total_removed
    
    def process(self):
        path = self.path_var.get().strip()
        
        if not path:
            messagebox.showwarning("警告", "请先选择文件或目录！")
            return
        
        if not os.path.exists(path):
            messagebox.showerror("错误", f"路径不存在: {path}")
            return
        
        self.log("=" * 60)
        self.log(f"开始处理: {path}")
        self.log("=" * 60)
        
        self.status_var.set("处理中...")
        self.progress_var.set(0)
        self.root.update()
        
        try:
            if os.path.isfile(path):
                processed_count, total_removed = self.process_file(path)
            elif os.path.isdir(path):
                processed_count, total_removed = self.process_directory(path)
            
            self.log("=" * 60)
            self.log(f"处理完成! 共处理 {processed_count} 个文件")
            self.log(f"总计移除 {total_removed} 个空行")
            self.log("=" * 60)
            
            self.status_var.set(f"完成! 处理了 {processed_count} 个文件")
            
            self.add_to_history(path, processed_count, total_removed)
            
            messagebox.showinfo("完成", f"处理完成!\n共处理 {processed_count} 个文件\n总计移除 {total_removed} 个空行")
            
        except Exception as e:
            self.log(f"处理出错: {str(e)}")
            self.status_var.set("处理出错")
            messagebox.showerror("错误", f"处理出错: {str(e)}")

def main():
    root = tk.Tk()
    app = RemoveEmptyLinesApp(root)
    root.mainloop()

if __name__ == '__main__':
    main()
